//---------------------------------------------------------------------------
///	@file
///		AspinaHand
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2020 - SHINANO-KENSHI Co,Ltd
///	@author
///		7022
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ASPINA_HAND_HEADER_INCLUDE
#define ASPINA_HAND_HEADER_INCLUDE

#include <ros/ros.h>
#include <sstream>
#include <unistd.h>

#include <time.h>

#include <aspina_hand/Hand.h>	// ハンド制御サービス・メッセージ
#include <aspina_hand/AspinaHandCommand.h>	// ハンド制御コマンド・メッセージ
#include <aspina_hand/AspinaHandStatus.h>	// ハンド制御ステータス・メッセージ

//---------------------------------------------------------
/// @brief ROSのサービスを使ったハンド制御
//---------------------------------------------------------
/// @note サービスによる制御はサービスの実行を完了するまで関数が終了しません。
//---------------------------------------------------------
class AspinaHandService
{
public:
	AspinaHandService(ros::NodeHandle *node)
	{
		mNode = node;
		mClient = mNode->serviceClient<aspina_hand::Hand>("aspina_hand/hand");
	}
	~AspinaHandService(void)
	{
		//nothing
	}

private:
	ros::NodeHandle *mNode;
	ros::ServiceClient mClient;

public:
	/// @brief ステータス
	typedef struct {
		bool stalled;      //!< true : ハンド異常
		bool reached_goal; //!< true : 動作完了（待機中）
		uint32_t position; //!< 現在位置[0.1%]
		uint32_t effort;   //!< 出力（力）[0.1%]
	} STATUS;

private:
	bool CheckID(int id)
	{
		if (id < 1 || id > 254){
			return false;
		}
		else{
			return true;
		}
	}
	uint32_t getTick()
	{
		struct timespec ts;
		unsigned theTick = 0U;

		clock_gettime( CLOCK_REALTIME, &ts);
		theTick = ts.tv_nsec / 1000000;
		theTick += ts.tv_sec * 1000;

		return theTick;
	}

public:
	//---------------------------------------------------------
	/// @brief 状態取得
	//---------------------------------------------------------
	/// @param id		ハンドID(1~31)
	/// @param status	ステータス格納先
	//---------------------------------------------------------
	/// @retval true	正常終了
	//---------------------------------------------------------
	bool GetStatus(int id, STATUS *status)
	{
		aspina_hand::Hand srv;

		if (CheckID(id) == false){
			return false;
		}
		srv.request.id = id;

		srv.request.cmd = 1;	// Status read
		if (mClient.call(srv) != 0){
			if (srv.response.stalled == 0){
				status->stalled = false;
			}
			else{
				status->stalled = true;
			}
			if (srv.response.reached_goal == 0){
				status->reached_goal = false;
			}
			else{
				status->reached_goal = true;
			}
			status->position = srv.response.position;
			status->effort = srv.response.effort;
			return true;
		}
		else{
			return false;
		}
	}
	//---------------------------------------------------------
	/// @brief アクション（動作）実行
	//---------------------------------------------------------
	/// @param id		ハンドID(1~31)
	/// @param position	指令位置[0.1%] 0:全開/1000:全閉
	/// @param effort	力[0.1%] 5:最小/1000:最大
	/// @param runTime	動作時間[ミリ秒] 250:最短/10000:最長
	/// @param wait		ハンド動作完了待ち false:待たない/true:待つ
	//---------------------------------------------------------
	/// @retval true	正常終了
	//---------------------------------------------------------
	/// @note wait=falseで実行する場合は、GetStatusでハンドの動作が @n
	///			完了していること(stalled=false,reached_goal=true)を @n
	///			確認してからActionを実行してください。
	//---------------------------------------------------------
	bool Action(int id, uint32_t position, uint32_t effort, uint32_t runTime, bool wait = false)
	{
		aspina_hand::Hand srv;

		if (CheckID(id) == false){
			return false;
		}
		srv.request.id = id;

		if (position < 0 || position > 1000 || effort < 5 || effort > 1000 || runTime < 250 || runTime > 10000){
			return false;
		}

		srv.request.cmd = 0;
		srv.request.position = position;
		srv.request.effort = effort;
		srv.request.runtime = runTime;
		if (mClient.call(srv) != 0){
			if (wait == true){
				struct timespec time;
				time.tv_sec = 0;
				time.tv_nsec = 10e-3 / 1e-9;
				STATUS status;
				uint32_t tick = getTick();
				uint32_t timeout = (uint32_t) (runTime * 1.1);
				for (;;){
					if (GetStatus(id, &status) == false){
						return false;
					}
					else{
						if (status.stalled == true){
							return false;
						}
						else{
							if (status.reached_goal == true){
								break;
							}
							else{
								nanosleep(&time, NULL);
								if ((getTick() - tick) >= timeout){
									return false;
								}
								else{
									// do nothing
								}
							}
						}
					}
				}
			}
			return true;
		}
		else{
			return false;
		}
	}
};

//---------------------------------------------------------
/// @brief ROSのトピックを使ったハンド制御
//---------------------------------------------------------
/// @note 実行完了を待たない場合、関数は即座に終了しますが、@n
/// 	  目的の動作を完了するまで関数を呼び続ける必要があります。
//---------------------------------------------------------
class AspinaHandTopic
{
public:
	AspinaHandTopic(ros::NodeHandle *node, int handNumber)
	{
		mNode = node;
		mHandNumber = handNumber;
		mState = new int[mHandNumber];
		for (int i = 0; i < mHandNumber; i++){
			mState[i] = 0;
		}
		mHandStatus = new HAND_STATUS[mHandNumber];
		for (int i = 0; i < mHandNumber; i++){
			mHandStatus[i].available = false;
		}
		mPub = mNode->advertise<aspina_hand::AspinaHandCommand>("aspina_hand/cmd", 1);
		mSub = mNode->subscribe("aspina_hand/status", 1, &AspinaHandTopic::HandCtrlCallback, this);
		mTimer = new uint32_t[mHandNumber];
	}
	~AspinaHandTopic(void)
	{
		//nothing
	}

private:
	typedef struct {
		bool available;
		;
		aspina_hand::AspinaHandStatus status;
	} HAND_STATUS;

private:
	int *mState;
	ros::NodeHandle *mNode;
	int mHandNumber;
	uint32_t *mTimer;
	HAND_STATUS *mHandStatus;
	ros::Publisher mPub;
	ros::Subscriber mSub;

public:
	/// @brief ステータス
	typedef struct {
		bool stalled;      //!< true : ハンド異常
		bool reached_goal; //!< true : 動作完了（待機中）
		uint32_t position; //!< 現在位置[0.1%]
		uint32_t effort;   //!< 出力（力）[0.1%]
	} STATUS;

private:
	void HandCtrlCallback(const ros::MessageEvent<aspina_hand::AspinaHandStatus const> &event)
	{
		int id = event.getMessage()->id;
		if (id >= 1 && id <= mHandNumber){
			HAND_STATUS *status = &mHandStatus[id - 1];
			status->status.position = event.getMessage()->position;
			status->status.effort = event.getMessage()->effort;
			status->status.result = event.getMessage()->result;
			status->status.reached_goal = event.getMessage()->reached_goal;
			status->status.stalled = event.getMessage()->stalled;
			status->available = true;
		}
		else{
			// do nothing
		}
	}

private:
	bool CheckID(int id)
	{
		if (id < 1 || id > mHandNumber){
			return false;
		}
		else{
			return true;
		}
	}
	uint32_t getTick()
	{
		struct timespec ts;
		unsigned theTick = 0U;

		clock_gettime( CLOCK_REALTIME, &ts);
		theTick = ts.tv_nsec / 1000000;
		theTick += ts.tv_sec * 1000;

		return theTick;
	}

public:
	//---------------------------------------------------------
	/// @brief 準備
	//---------------------------------------------------------
	/// @retval true	正常
	//---------------------------------------------------------
	/// @note GetStatusなど実行する前に本関数を一度実行してください。
	//---------------------------------------------------------
	bool Startup(int id)
	{
		if (CheckID(id) == false){
			return false;
		}

		int i;
		for (i = 0; i < 5; i++){
			if (GetStatus(id, NULL, true) == 0){
				break;
			}
		}

		if (i < 5){
			return true;
		}
		else{
			return false;
		}
	}
	//---------------------------------------------------------
	/// @brief 状態取得
	//---------------------------------------------------------
	/// @param id		ハンドID(1~31)
	/// @param status	ステータス格納先
	/// @param wait		取得完了待ち(false:待たない)
	//---------------------------------------------------------
	/// @retval 0	取得完了
	/// @retval 1	取得待ち
	/// @retval -1	取得エラー
	//---------------------------------------------------------
	int GetStatus(int id, STATUS *status, bool wait = false)
	{
		if (CheckID(id) == false){
			return false;
		}

		id--;

		aspina_hand::AspinaHandCommand cmd;
		if (wait == false){
			if (mState[id] == 0){
				cmd.id = id + 1;
				cmd.cmd = 1;
				mPub.publish(cmd);
				mState[id] = 1;
				mTimer[id] = getTick();
			}
			else{
				if (mState[id] == 1){
					if (mHandStatus[id].available == true){
						aspina_hand::AspinaHandStatus *s = &mHandStatus[id].status;
						if (status != NULL){
							status->stalled = s->stalled;
							status->reached_goal = s->reached_goal;
							status->position = s->position;
							status->effort = s->effort;
						}
						mHandStatus[id].available = false;
						mState[id] = 0;
						return 0;
					}
					else{
						if ((getTick() - mTimer[id]) >= 500){
							mState[id] = 0;
							return -1;
						}
						else{
							// do nothing
						}
					}
				}
				else{
					// do nothing
				}
			}
			return 1;
		}
		else{
			if (mState[id] == 0){
				cmd.id = id + 1;
				cmd.cmd = 1;
				mPub.publish(cmd);
			}
			struct timespec time;
			time.tv_sec = 0;
			time.tv_nsec = 1e-3 / 1e-9;
			uint32_t timer = getTick();
			while (mHandStatus[id].available == false){
				if ((getTick() - timer) >= 500){
					mState[id] = 0;
					return -1;
				}
				ros::spinOnce();
				nanosleep(&time, NULL);
			}
			aspina_hand::AspinaHandStatus *s = &mHandStatus[id].status;
			if (status != NULL){
				status->stalled = s->stalled;
				status->reached_goal = s->reached_goal;
				status->position = s->position;
				status->effort = s->effort;
			}
			mHandStatus[id].available = false;
			mState[id] = 0;
			return 0;
		}
	}
	//---------------------------------------------------------
	/// @brief アクション（動作）実行
	//---------------------------------------------------------
	/// @param id		ハンドID(1~31)
	/// @param position	指令位置[0.1%] 0:全開/1000:全閉
	/// @param effort	力[0.1%] 5:最小/1000:最大
	/// @param runTime	動作時間[ミリ秒] 250:最短/10000:最長
	/// @param wait		ハンド動作完了待ち false:待たない/true:待つ
	//---------------------------------------------------------
	/// @retval true	正常終了
	//---------------------------------------------------------
	/// @note wait=falseで実行する場合は、GetStatusでハンドの動作が @n
	///			完了していること(stalled=false,reached_goal=true)を @n
	///			確認してからActionを実行してください。
	//---------------------------------------------------------
	bool Action(int id, uint32_t position, uint32_t effort, uint32_t runTime, bool wait = false)
	{
		if (CheckID(id) == false){
			return false;
		}

		if (position < 0 || position > 1000 || effort < 5 || effort > 1000 || runTime < 250 || runTime > 10000){
			return false;
		}

		id--;

		aspina_hand::AspinaHandCommand cmd;
		cmd.id = id + 1;
		cmd.cmd = 0;
		cmd.position = position;
		cmd.effort = effort;
		cmd.runtime = runTime;
		mPub.publish(cmd);

		STATUS sts;
		mState[id] = 1;	// Actionコマンドで状態トピックが送信されてくるので、GetStatus関数を呼んだ時にステータス取得トピックを送信しないようにする。
		if (GetStatus(id + 1, &sts, true) != 0){
			return false;
		}
		if (sts.stalled == true){
			return false;
		}

		if (wait == true){
			uint32_t timer = getTick();
			runTime = (uint32_t) ((double) runTime * 1.5);
			for (;;){
				if (GetStatus(id + 1, &sts, true) != 0){
					return false;
				}
				else{
					if (sts.stalled == true){
						return false;
					}
					else{
						if (sts.reached_goal == true){
							break;
						}
						else{
							if ((getTick() - timer) >= runTime){
								return false;
							}
							else{
								// do nothing
							}

						}
					}
				}
			}
			return true;
		}
		else{
			return true;
		}
	}
};

#endif // ASPINA_HAND_HEADER_INCLUDE

