//---------------------------------------------------------------------------
///	@file
///		HandModbus.cpp
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2019 - SHINANO-KENSHI Co,Ltd
///	@author
///		7049
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "HandModbus.h"

HandModbus::HandModbus(ISerial* serial) : mSlaveID(1)
{
	mModbus = new Modbus(serial, 100);
}

HandModbus::~HandModbus(void)
{
	delete mModbus;
}

INT HandModbus::open(const char* portName, INT baudRate, INT dataBits, INT parity, INT stopBits)
{
	if (mModbus->Open(portName, baudRate, dataBits, parity, stopBits) != false){
		return 0;
	}
	else{
		return 1;
	}
}

INT HandModbus::close(void)
{
	return mModbus->Close();
}

BOOL HandModbus::isOpen(void)
{
	return mModbus->isOpen();
}

INT HandModbus::setSlaveID(INT id)
{
	mSlaveID = (UINT8)id;

	return 0;
}

INT HandModbus::sendPortNumbers(int* dataDio)
{
	if (dataDio == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0107;
	UINT16 data[4];

	data[0] = (UINT16)dataDio[0];
	data[1] = (UINT16)dataDio[1];
	data[2] = (UINT16)dataDio[2];
	data[3] = (UINT16)dataDio[3];

	if (mModbus->WriteMultipleRegister(id, adrs, 4, data) != false){
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvPortNumbers(int* dataDio)
{
	if (dataDio == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0107;
	UINT16 data[4];

	if (mModbus->ReadHoldingRegister(id, adrs, 4, data) != false){
		//
		dataDio[0] = (int)data[0];
		dataDio[1] = (int)data[1];
		dataDio[2] = (int)data[2];
		dataDio[3] = (int)data[3];
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::sendHandServoOff(void)
{
	//TODO: Write Single Register (adrs = REG_ADRS_OPERATION)

	return 0;
}

INT HandModbus::sendHandServoOn(void)
{
	//TODO: Write Single Register (adrs = REG_ADRS_OPERATION)

	return 0;
}

INT HandModbus::sendHandStart(INT num)
{
	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0010;
	UINT16 code = (UINT16)num;

	if (mModbus->WriteSingleRegister(id, adrs, &code) != false){
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::sendHandStop(void)
{
	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0010;
	UINT16 code = 0x00A0;

	if (mModbus->WriteSingleRegister(id, adrs, &code) != false){
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::sendHandAlarmReset(void)
{
	//TODO: Write Single Register (adrs = REG_ADRS_RESET_ALARM)

	return 0;
}

INT HandModbus::recvHandAlarm(UINT32* bits)
{
	if (bits == NULL){
		return -1;
	}

	//TODO: Read Input Register (adrs = REG_ADRS_ALARM_BIT_STATUS_H)

	return 0;
}

INT HandModbus::recvHandWarning(UINT32* bits)
{
	if (bits == NULL){
		return -1;
	}

	//TODO: Read Input Register (adrs = REG_ADRS_WARNING_BIT_STATUS_H)

	return 0;
}

INT HandModbus::recvHandStatus(UINT32* bits)
{
	if (bits == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0024;
	UINT16 data[2];

	if (mModbus->ReadInputRegister(id, adrs, 2, data) != false){
		//
		*bits = (UINT32)data[0] << 16;
		*bits += (UINT32)data[1];
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvHandCurrentPosition(INT* posPermill)
{
	if (posPermill == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0028;
	UINT16 data[1];

	if (mModbus->ReadInputRegister(id, adrs, 1, data) != false){
		//
		*posPermill = (INT16)data[0];
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvHandCurrentTorque(INT* trqPermill)
{
	if (trqPermill == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0026;
	UINT16 data[1];

	if (mModbus->ReadInputRegister(id, adrs, 1, data) != false){
		//
		*trqPermill = (INT16)data[0];
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvHandCurrentThermistor(INT* celsius)
{
	if (celsius == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x002A;
	UINT16 data[1];

	if (mModbus->ReadInputRegister(id, adrs, 1, data) != false){
		//
		*celsius = (INT16)data[0];
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvHandCurrentEncoder(INT* encPulse)
{
	if (encPulse == NULL){
		return -1;
	}

	UINT8 id = mSlaveID;
	UINT16 adrs = 0x300A;
	UINT16 data[2];
	UINT32 enc;

	if (mModbus->ReadHoldingRegister(id, adrs, 2, data) != false){
		//
		enc = (UINT32)data[0] << 16;
		enc += (UINT32)data[1];
		//
		*encPulse = (INT)((INT32)enc);
		//
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::sendParamPositioning(INT num, const HandParamPartial* pParam)
{
	UINT8 id = mSlaveID;
	UINT16 adrs = (UINT16)(0x1000 + (num * 0x0100));
	UINT16 data[10];

	data[0] = (UINT16)pParam->orderPos;
	data[1] = (UINT16)pParam->orderMsec;
	data[2] = (UINT16)pParam->orderTrq;
	data[3] = (UINT16)pParam->pushWidth;
	data[4] = (UINT16)pParam->pushSpd;
	data[5] = (UINT16)pParam->pushPow;
	data[6] = (UINT16)pParam->areaSigLower;
	data[7] = (UINT16)pParam->areaSigUpper;
	data[8] = (UINT16)pParam->gripErrLower;
	data[9] = (UINT16)pParam->gripErrUpper;

	if (mModbus->WriteMultipleRegister(id, adrs, 10, data) != false){
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}

INT HandModbus::recvParamPositioning(INT num, const HandParamPartial* pParam)
{
	//TODO: Read Registers

	return 0;
}

INT HandModbus::sendMemoryWrite(INT accessID)
{
	UINT8 id = mSlaveID;
	UINT16 adrs = 0x0050;
	UINT16 code = (UINT16)accessID;

	if (mModbus->WriteSingleRegister(id, adrs, &code) != false){
		return 0;//ok
	}
	else{
		return 1;//ng
	}
}
