//---------------------------------------------------------------------------
///	@file
///		ISerial
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2019 - SHINANO-KENSHI Co,Ltd
///	@author
///		7049
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef	ISERIAL_HEADER_INCLUDE
#define	ISERIAL_HEADER_INCLUDE

#include "shinanodef.h"

//---------------------------------------------------------------------------
/// @brief	dataStructure
//---------------------------------------------------------------------------
typedef struct {
	UINT32	dwBytes;
	void	*pAddr;
} SERIAL_BUF_INFO;

//---------------------------------------------------------------------------
/// @brief	serialCommErrorValues
//---------------------------------------------------------------------------
enum tag_SERIAL_ERROR {
	SERIAL_ERROR_SUCCESS = 0,
	SERIAL_ERROR_TIMEOUT,
	SERIAL_ERROR_PARAMETER,
	SERIAL_ERROR_OPEN,
	SERIAL_ERROR_NOT_OPEN,
	SERIAL_ERROR_OVERRUN,
	SERIAL_ERROR_NOISE,
	SERIAL_ERROR_FRAMING,
	SERIAL_ERROR_PARITY,
	SERIAL_ERROR_BUSY,
	SERIAL_ERROR_UNSUPPORTED
};

//---------------------------------------------------------------------------
/// @brief	serialCommunicationInterface
//---------------------------------------------------------------------------
class ISerial
{
public:
	virtual ~ISerial(void)
	{
	}

public:
	//---------------------------------------------------------------------------
	/// @brief	Initialize
	//---------------------------------------------------------------------------
	/// @param	none
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	/// @note	It is executed once when the port is registered.
	//---------------------------------------------------------------------------
	virtual INT Init(void) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Open serial port
	//---------------------------------------------------------------------------
	/// @param	port     portName
	/// @param	baudrate baudRate[bps]
	/// @param	dataBits dataBitLength
	/// @param	stopBits stopBit(1,2,else)
	/// @param	parity   parityBit(0:none,1:even,2:odd)
	/// @param	echoBack echoBack(0:none,1:autoCancel)
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	/// @note	If you want to reopen, close it once.
	//---------------------------------------------------------------------------
	virtual INT Open(const char* port, INT32 baudrate, INT dataBits, INT stopBits, INT parity, INT echoBack) = 0;

	//---------------------------------------------------------------------------
	/// @brief	serial setting (extend)
	//---------------------------------------------------------------------------
	/// @param	pdwParam parameterStartAddress
	/// @param	dwMask   Specify the parameter that enables the setting with a bit
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	/// @note	Extended parameters depend on the system. @n
	///			In dwMask, bit 0 indicates parameter 0, and a total of 32 types can be specified.
	//---------------------------------------------------------------------------
	virtual INT SettingEx(const UINT32 *pdwParam,UINT32 dwMask) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Close serial port
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	virtual INT Close(void) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Send
	//---------------------------------------------------------------------------
	/// @param	data    address of send data
	/// @param	bytes   number of send data bytes
	/// @param	timeout timeoutTime[msec]
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	/// @note	If you specify 0 for timeout, the function ends immediately after the start of transmission, @n
	///			and you can check the transmission status with SendCheck (effective when transmitting in the background). @n
	///			If 0 is specified for timeout and this function is executed again with timeout = 0 during transmission, @n
	///			a timeout error will occur (if transmission is completed before executing this function, @n
	///			no error will occur and the next transmission will start. NS).
	//---------------------------------------------------------------------------
	virtual INT Send(const void *data, INT bytes, UINT32 timeout) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Check send status
	//---------------------------------------------------------------------------
	/// @retval	sendStatus (0:ready, 1:busy）
	//---------------------------------------------------------------------------
	virtual INT SendCheck(void) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Check recieve status
	//---------------------------------------------------------------------------
	/// @retval	Returns the number of bytes of data stored in the receive buffer.
	//---------------------------------------------------------------------------
	virtual INT ReceiveCheck(void) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Receive
	//---------------------------------------------------------------------------
	/// @param	data    address of send data
	/// @param	bytes   number of send data bytes
	/// @param	timeout timeoutTime[msec]
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	virtual INT Receive(void* data, INT bytes, UINT32 timeout) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Clear buffer
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	virtual INT Clear(void) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Termination setting
	//---------------------------------------------------------------------------
	/// @param	iOn (0:Off, 1:On)
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	virtual INT Termination(INT iOn) = 0;

	//---------------------------------------------------------------------------
	/// @brief	Communication direction control (used for half-duplex communication)
	//---------------------------------------------------------------------------
	/// @param	iDir commDirection(SERIAL_DIR_SEND/SERIAL_DIR_RECEIVE)
	//---------------------------------------------------------------------------
	/// @retval	commErrorCode(SERIAL_ERROR_xxxx)
	//---------------------------------------------------------------------------
	virtual INT Dir(INT iDir){return SERIAL_ERROR_SUCCESS;};
};

#endif	// ISERIAL_HEADER_INCLUDE
