//---------------------------------------------------------------------------
///	@file
///		Modbus
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2019 - SHINANO-KENSHI Co,Ltd
///	@author
///		7022
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef MODBUS_H_
#define MODBUS_H_

#include <iostream>
#include <pthread.h>
#include "ISerial.h"

class Modbus
{

private:
	typedef struct {
		UINT8	*message;
		INT		len;
	} MSGS;

private:
	bool mOpen;
	ISerial	*sp;
	MSGS *mBuffer;
	pthread_mutex_t mMutex;

public:
//	Modbus(ISerial *comm, int max_register, bool open)
	Modbus(ISerial *comm, int max_register)
	{
		sp = comm;
		mOpen = false;
		mBuffer = new MSGS;
		mBuffer->message = new UINT8[max_register * 2 + 32];
		mMutex = PTHREAD_MUTEX_INITIALIZER;
		pthread_mutex_init(&mMutex,NULL);
	}
	virtual ~Modbus()
	{
		if(mBuffer != NULL){
			delete	mBuffer->message;
			delete	mBuffer;
			mBuffer = NULL;
		}
		pthread_mutex_destroy(&mMutex);
	}

private:
	void Lock()
	{
		pthread_mutex_lock(&mMutex);
	}
	void Unlock()
	{
		pthread_mutex_unlock(&mMutex);
	}

public:
//	bool Open(int baudRate,int databits,int parity,int stopBits)
	bool Open(const char* portName, int baudRate, int databits, int parity, int stopBits)
	{
		bool ret = true;

		Lock();

		//Ensure port isn't already opened:
		if (mOpen == false) {
			//Assign desired settings to the serial port:
//			if(sp->Open(baudRate,databits,stopBits,parity,0) == 0){
			if(sp->Open(portName, baudRate, databits, parity, stopBits, 0) == 0){
				mOpen = true;
			} else {
				ret = false;
			}
		}

		Unlock();

		return ret;
	}

	bool Close()
	{
		bool ret = true;

		Lock();

		//Ensure port is opened before attempting to close:
		if (mOpen != false) {
			mOpen = false;
			sp->Close();
		}

		Unlock();

		return ret;
	}

	bool isOpen()
	{
		Lock();

		bool ret = mOpen;

		Unlock();

		return ret;
	}

private:
	void GetCRC(const MSGS *message,UINT16 *crc)
	{
		//Function expects a modbus message of any length as well as a 2 byte CRC array in which to
		//return the CRC values:

		UINT16 CRCFull = 0xFFFF;
		char CRCLSB;

		for (int i = 0;i < message->len - 2;i++) {
			CRCFull = (UINT16)(CRCFull ^ message->message[i]);

			for (int j = 0;j < 8;j++) {
				CRCLSB = (char)(CRCFull & 0x0001);
				CRCFull = (UINT16)((CRCFull >> 1) & 0x7FFF);

				if (CRCLSB == 1)
					CRCFull = (UINT16)(CRCFull ^ 0xA001);
			}
		}

		*crc = CRCFull;
	}

	void BuildMessage(UINT8 address,UINT8 type,UINT16 start,UINT16 registers,MSGS *message)
	{
		if (type == 6){
			//write single register
			BuildMessageS(address, type, start, 1, message);
		}
		else{
			//write multi registers, read input/holding registers
			BuildMessageL(address, type, start, registers, message);
		}
	}

	void BuildMessageS(UINT8 address,UINT8 type,UINT16 start,UINT16 registers,MSGS *message)
	{
		//Array to receive CRC bytes:
		UINT16	crc;

		message->message[0] = address;
		message->message[1] = type;
		message->message[2] = (UINT8)(start >> 8);
		message->message[3] = (UINT8)start;

		GetCRC((const MSGS *)message,&crc);
		message->message[message->len - 2] = (UINT8)(crc & 0xFF);
		message->message[message->len - 1] = (UINT8)(crc >> 8);
	}

	void BuildMessageL(UINT8 address,UINT8 type,UINT16 start,UINT16 registers,MSGS *message)
	{
		//Array to receive CRC bytes:
		UINT16	crc;

		message->message[0] = address;
		message->message[1] = type;
		message->message[2] = (UINT8)(start >> 8);
		message->message[3] = (UINT8)start;
		message->message[4] = (UINT8)(registers >> 8);
		message->message[5] = (UINT8)registers;

		GetCRC((const MSGS *)message,&crc);
		message->message[message->len - 2] = (UINT8)(crc & 0xFF);
		message->message[message->len - 1] = (UINT8)(crc >> 8);
	}

	bool CheckResponse(const MSGS *response)
	{
		//Perform a basic CRC check:
		UINT16	crc;
		GetCRC(response,&crc);
		if ((UINT8)(crc & 0xFF) == response->message[response->len - 2] && (UINT8)(crc >> 8) == response->message[response->len - 1])
			return true;
		else
			return false;
	}

	bool GetResponse(MSGS *response)
	{
		//There is a bug in .Net 2.0 DataReceived Event that prevents people from using this
		//event as an interrupt to handle data (it doesn't fire all of the time).  Therefore
		//we have to use the ReadByte command for a fixed length as it's been shown to be reliable.
		return sp->Receive(response->message,response->len,300) == 0 ? true : false;
	}

public:
	// value is BigEndian
	bool WriteRegister(UINT8 address,UINT8 fc,UINT16 start,UINT16 registers,const UINT16 *values)
	{
		if (fc == 6){
			return WriteSingleRegister(address, start, values);
		}
		else{
			return WriteMultipleRegister(address, start, registers, values);
		}
	}

	bool WriteSingleRegister(UINT8 address,UINT16 start,const UINT16 *values)
	{
		bool ret = true;

		Lock();

		//Ensure port is open:
		if (mOpen == true) {
			//Clear in/out buffers:
			sp->Clear();
			//Add bytecount to message:
			mBuffer->len = 8;
			//Put write values into message prior to sending:
			mBuffer->message[4] = (UINT8)(*values >> 8);
			mBuffer->message[5] = (UINT8)(*values);
			//Build outgoing message:
			BuildMessage(address,6,start,1,mBuffer);
			//Send Modbus message to Serial Port:
			if(sp->Send(mBuffer->message,mBuffer->len,300) == 0){
				mBuffer->len = 8;
				if(GetResponse(mBuffer) == false){
					ret = false;
				} else {
					if (CheckResponse(mBuffer) == false) {
						ret = false;
					} else {
						ret = true;
					}
				}
			} else {
				ret = false;
			}
		} else {
			ret =  false;
		}

		Unlock();

		return ret;
	}

	bool WriteMultipleRegister(UINT8 address,UINT16 start,UINT16 registers,const UINT16 *values)
	{
		bool ret = true;

		Lock();

		//Ensure port is open:
		if (mOpen == true) {
			//Clear in/out buffers:
			sp->Clear();
			//Add bytecount to message:
			mBuffer->len = 9 + 2 * registers;
			mBuffer->message[6] = (UINT8)(registers * 2);
			//Put write values into message prior to sending:
			for (UINT16 i = 0;i < registers;i++) {
				mBuffer->message[7 + 2 * i] = (UINT8)(values[i] >> 8);
				mBuffer->message[8 + 2 * i] = (UINT8)(values[i]);
			}
			//Build outgoing message:
			BuildMessage(address,16,start,registers,mBuffer);
			//Send Modbus message to Serial Port:
			if(sp->Send(mBuffer->message,mBuffer->len,300) == 0){
				mBuffer->len = 8;
				if(GetResponse(mBuffer) == false){
					ret = false;
				} else {
					if (CheckResponse(mBuffer) == false) {
						ret = false;
					} else {
						ret = true;
					}
				}
			} else {
				ret = false;
			}
		} else {
			ret =  false;
		}

		Unlock();

		return ret;
	}

	// value is BigEndian
	bool ReadRegister(UINT8 address,UINT8 fc,UINT16 start,UINT16 registers,UINT16 *values)
	{
		bool ret = true;

		Lock();

		//Ensure port is open:
		if (mOpen == true) {
			//Clear in/out buffers:
			sp->Clear();
			//Build outgoing modbus message:
			mBuffer->len = 8;
			BuildMessage(address,fc,start,registers,mBuffer);
			//Send modbus message to Serial Port:
			if(sp->Send(mBuffer->message,mBuffer->len,300) == 0){
				mBuffer->len = 5 + 2 * registers;
				if(GetResponse(mBuffer) == true){
					if (CheckResponse(mBuffer) == true) {
						//Return requested register values:
						for (int i = 0;i < (mBuffer->len - 5) / 2;i++) {
							UINT16	data = (UINT16)mBuffer->message[2 * i + 3];
							data <<= 8;
							data += (UINT16)mBuffer->message[2 * i + 4];
							*values++ = data;
						}
					} else {
						ret = false;
					}
				} else {
					ret = false;
				}
			} else {
				ret = false;
			}
		} else {
			ret = false;
		}

		Unlock();

		return ret;
	}

	bool ReadHoldingRegister(UINT8 address, UINT16 start, UINT16 registers, UINT16 *values)
	{
		return ReadRegister(address, 3, start, registers, values);
	}

	bool ReadInputRegister(UINT8 address, UINT16 start, UINT16 registers, UINT16 *values)
	{
		return ReadRegister(address, 4, start, registers, values);
	}

//	// debug
//	void dump(const MSGS *msg)
//	{
//		for(int i = 0;i < msg->len;i++){
//			monprintf("%2d : %02X\r\n",i,msg->message[i]);
//		}
//	}

};

#endif /* MODBUS_H_ */
