//---------------------------------------------------------------------------
///	@file
///		RobotARH.cpp
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2019 - SHINANO-KENSHI Co,Ltd
///	@author
///		7049
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <iostream>
#include <time.h>
#include "RobotARH.h"

RobotARH::RobotARH(void)
	: mIHC(NULL)
	, mNameFd("")
	, mBaudrate(115200)
	, mSlaveID(1)
	, mNumDio1(HandDioCode::iSEL0)
	, mNumDio2(HandDioCode::iALMRST)
	, mNumDio3(HandDioCode::iGRIPERR)
	, mNumDio4(HandDioCode::iALARM)
	, mCParamPart()
	, mClockStart(0)
	, mClockNow(0)
{

}

RobotARH::~RobotARH(void)
{

}

void RobotARH::setIHandComm(IHandComm* pI)
{
	mIHC = pI;
}

INT RobotARH::openComm(std::string fdname)
{
	INT state = stateNG;

	mNameFd = fdname;

	if (mIHC != NULL){
		//
		if (mIHC->isOpen()){
			mIHC->close();
		}
		//
		state = mIHC->open(fdname.c_str(), mBaudrate, 8, 2, 1);
	}

	return state;
}

INT RobotARH::setBaudrate(INT baudrate)
{
	mBaudrate = baudrate;

	return 0;
}

INT RobotARH::setSlaveID(INT slaveid)
{
	INT state = stateNG;

	mSlaveID = slaveid;

	if (mIHC != NULL){
		//
		state = mIHC->setSlaveID(slaveid);
	}

	return state;
}

std::string RobotARH::getCommPortName(void)
{
	return mNameFd;
}

INT RobotARH::getBaudrate(void)
{
	return mBaudrate;
}

INT RobotARH::getSlaveID(void)
{
	return mSlaveID;
}

INT RobotARH::loadPorts(void)
{
	INT state = stateNG;

	if (mIHC != NULL){
		//
		int dataDio[4];
		//
		if ((state = mIHC->recvPortNumbers(dataDio)) == 0){
			mNumDio1 = dataDio[0];
			mNumDio2 = dataDio[1];
			mNumDio3 = dataDio[2];
			mNumDio4 = dataDio[3];
		}
	}

	return state;
}

INT RobotARH::savePorts(void)
{
	int state = stateNG;

	if (mIHC != NULL){
		//
		int dataDio[4];
		//
		dataDio[0] = mNumDio1;
		dataDio[1] = mNumDio2;
		dataDio[2] = mNumDio3;
		dataDio[3] = mNumDio4;
		//
		state = mIHC->sendPortNumbers(dataDio);
	}

	return state;
}

INT RobotARH::getDio1(void) const
{
	return mNumDio1;
}

INT RobotARH::getDio2(void) const
{
	return mNumDio2;
}

INT RobotARH::getDio3(void) const
{
	return mNumDio3;
}

INT RobotARH::getDio4(void) const
{
	return mNumDio4;
}

void RobotARH::setDio1(INT portnum)
{
	mNumDio1 = portnum;
}

void RobotARH::setDio2(INT portnum)
{
	mNumDio2 = portnum;
}

void RobotARH::setDio3(INT portnum)
{
	mNumDio3 = portnum;
}

void RobotARH::setDio4(INT portnum)
{
	mNumDio4 = portnum;
}

void RobotARH::setHandPos(INT position)
{
	mCParamPart.orderPos = position;
}

void RobotARH::setHandTime(INT msec)
{
	mCParamPart.orderMsec = msec;
}

void RobotARH::setHandTrq(INT torque)
{
	mCParamPart.orderTrq = torque;
}

void RobotARH::setHandPosGripErrorLower(INT position)
{
	mCParamPart.gripErrLower = position;
}

void RobotARH::setHandPosGripErrorUpper(INT position)
{
	mCParamPart.gripErrUpper = position;
}

void RobotARH::setHandPushPos(INT position)
{
	mCParamPart.pushWidth = position;
}

void RobotARH::setHandPushSpd(INT speed)
{
	mCParamPart.pushSpd = speed;
}

void RobotARH::setHandPushTrq(INT torque)
{
	mCParamPart.pushPow = torque;
}

INT RobotARH::opeHand(INT opecode) const
{
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->sendHandStart(opecode);
	}

	return state;
}

INT RobotARH::updtHand(INT partnum) const
{
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->sendParamPositioning(partnum, &mCParamPart);
	}

	return state;
}

INT RobotARH::getHandStatus(UINT32* status) const
{
	INT state = stateNG;

	if (status != NULL){
		if (mIHC != NULL){
			state = mIHC->recvHandStatus(status);
		}
	}

	return state;
}

INT RobotARH::writeMemoryCommon(void) const
{
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->sendMemoryWrite(1); // 1:SaveCommon
	}

	return state;
}

INT RobotARH::getHandCurrentPosition(void) const
{
	INT pos = 0;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandCurrentPosition(&pos);
	}

	if (state != 0){
		pos = 0;
	}
	else{
		if (pos < 0){
			pos = 0;
		}
	}

	return pos;
}

INT RobotARH::getHandCurrentTorque(void) const
{
	INT trq = 0;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandCurrentTorque(&trq);
	}

	if (state != 0){
		trq = 0;
	}
	else{
		if (trq < 0){
			trq *= -1;
		}
	}

	return trq;
}

INT RobotARH::getHandCurrentThermistor(void) const
{
	INT temp = 0;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandCurrentThermistor(&temp);
	}

	if (state != 0){
		temp = 0;
	}

	return temp;
}

INT RobotARH::getHandCurrentEncoder(void) const
{
	INT enc = 0;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandCurrentEncoder(&enc);
	}

	if (state != 0){
		enc = 0;
	}

	return enc;
}

BOOL RobotARH::isHandStatusAlarm(void) const
{
	UINT32 bits = 0u;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandStatus(&bits);
	}

	if (state != 0){
		//nothing
	}
	else{
		//bit2:alarm
		if ((bits & HandStatusCode::bitAlarm) != 0){
			return true;
		}
	}

	return false;
}

BOOL RobotARH::isHandStatusGripError(void) const
{
	UINT32 bits = 0u;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandStatus(&bits);
	}

	if (state != 0){
		//nothing
	}
	else{
		//bit3:gripError
		if ((bits & HandStatusCode::bitGripErr) != 0){
			return true;
		}
	}

	return false;
}

BOOL RobotARH::isHandStatusReady(void) const
{
	UINT32 bits = 0u;
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->recvHandStatus(&bits);
	}

	if (state != 0){
		//nothing
	}
	else{
		//bit1:handReady
		if ((bits & HandStatusCode::bitReady) != 0){
			return true;
		}
	}

	return false;
}

void RobotARH::clearTimeout(void)
{
	mClockStart = (UINT32)clock();
}

BOOL RobotARH::isTimeout(UINT32 sec)
{
	UINT32 elasped; //unit:[sec]

	mClockNow = (UINT32)clock();

	elasped = (UINT32)(mClockNow - mClockStart) / CLOCKS_PER_SEC;
	if (elasped > sec){
		return true;
	}
	else{
		return false;
	}
}

void RobotARH::sleep(UINT32 msec)
{
	//NOTE:Thread.Sleep(<msec>)

	UINT32 elasped; //unit:[msec]

	clock_t clockPrev = clock();
	clock_t clockNow;

	while(1){
		clockNow = clock();
		elasped = (UINT32)((clockNow - clockPrev) * 1000 / CLOCKS_PER_SEC);
		if (elasped > msec){
			break;
		}
	}
}

INT RobotARH::moveHand(INT position, BOOL isWait)
{
	INT state = stateOK;

	if (state == stateOK){
		state = moveHandParamSet(position);
	}

	if (state == stateOK){
		state = moveHandStart();
	}

	if (state == stateOK){
		state = moveHandWait(isWait);
	}

	return state;
}

INT RobotARH::moveHand(INT position, INT torque, BOOL isWait)
{
	INT state = stateOK;

	if (state == stateOK){
		state = moveHandParamSet(position, torque);
	}

	if (state == stateOK){
		state = moveHandStart();
	}

	if (state == stateOK){
		state = moveHandWait(isWait);
	}

	return state;
}

INT RobotARH::stopHand(void)
{
	INT state = stateOK;

	if (state == stateOK){
		state = moveHandStop();
	}

	if (state == stateOK){
		state = moveHandWait(true);
	}

	return state;
}

INT RobotARH::moveHandParamSet(INT position)
{
	INT state = stateNG;
	INT partnum = movePartNum;

	HandParamPartial* pParams = &mCParamPart;
	BOOL isPush = this->isReqPushIn(pParams);

	//set position
	if (calcMoveTarget(pParams, position, isPush) != stateOK){
		return stateNG;
	}

	if (mIHC != NULL){
		state = mIHC->sendParamPositioning(partnum, pParams);
	}

	return state;
}

INT RobotARH::moveHandParamSet(INT position, INT torque)
{
	INT state = stateNG;
	INT partnum = movePartNum;

	HandParamPartial* pParams = &mCParamPart;
	BOOL isPush = this->isReqPushIn(pParams);

	//cache previous torque
	INT torquePrev = pParams->orderTrq;

	//set position
	if (calcMoveTarget(pParams, position, isPush) != stateOK){
		return stateNG;
	}

	//set torque
	pParams->orderTrq = torque;

	if (mIHC != NULL){
		state = mIHC->sendParamPositioning(partnum, pParams);
	}

	//write back previous torque
	pParams->orderTrq = torquePrev;

	return state;
}

INT RobotARH::moveHandStart(void)
{
	INT state = stateNG;
	INT opecode = HandOpeCode::opeHandMove + movePartNum;

	if (mIHC != NULL){
		state = mIHC->sendHandStart(opecode);
	}

	return state;
}

INT RobotARH::moveHandStop(void)
{
	INT state = stateNG;

	if (mIHC != NULL){
		state = mIHC->sendHandStop();
	}

	return state;
}

INT RobotARH::moveHandWait(BOOL isWait)
{
	INT state = stateNG;
	UINT32 bits = 0u;

	if (isWait == false){
		return stateOK;
	}

	this->clearTimeout();

	while(1){
		if (mIHC != NULL){
			state = mIHC->recvHandStatus(&bits);
		}

		if (state != stateOK){
			break; //comm failure
		}
		if ((bits & HandStatusCode::bitSrvOn) == 0){
			break; //servo off
		}
		if ((bits & HandStatusCode::bitReady) != 0){
			break; //hand ready
		}
		if ((bits & HandStatusCode::bitAlarm) != 0){
			break; //alarm detect
		}
		if (this->isTimeout(WaitTimeoutSec) != false){
			break;
		}

		this->sleep(WaitSleepMsec);
	}

	return stateOK;
}

INT RobotARH::calcMoveTarget(HandParamPartial* params, INT position, BOOL isPush)
{
	INT state = stateNG;

	if (isPush == false){
		//no push
		params->orderPos = position;
		return stateOK;
	}

	INT currPos = 0;
	INT pushPos = params->pushWidth;

	if (mIHC != NULL){
		state = mIHC->recvHandCurrentPosition(&currPos);
	}

	if (state == stateOK){
		if (currPos < position){
			//push(+):close
			if (pushPos < 0){
				pushPos *= -1;
			}
		}
		else if (currPos > position){
			//push(-):open
			if (pushPos > 0){
				pushPos *= -1;
			}
		}
		else{
			//push disable
			state = stateNG;
		}
	}

	if (state == stateOK){
		params->orderPos = position - pushPos;
		params->pushWidth = pushPos;
	}

	return state;
}

BOOL RobotARH::isReqPushIn(HandParamPartial* params)
{
	if ((params->pushWidth == 0)||(params->pushSpd == 0)||(params->pushPow == 0)){
		return false;
	}
	else{
		return true;
	}
}
