//---------------------------------------------------------------------------
///	@file
///		serialLinux.cpp
///	@brief
///		for ROS package
///	@attention
///		Copyright (C) 2019 - SHINANO-KENSHI Co,Ltd
///	@author
///		7049
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include "serialLinux.hpp"

static int Serial_Open(const char *port, long br);
static int Serial_Send(int port, const void *data, int bytes);
static int Serial_Receive(int port, void *data, int bytes, unsigned int timeout, SerialLinux::LoopCallback cb, void *param);
static void Serial_Close(int port);
static void Serial_Clear(int port);
static long msElapse(struct timespec *time, struct timespec *start);

SerialLinux::SerialLinux(void) : mSePort(0)
{
	mCB = NULL;
	mParam = NULL;
}

SerialLinux::~SerialLinux(void)
{
	//nothing
}

INT SerialLinux::Open(const char* port, INT32 baudrate, INT dataBits, INT stopBits, INT parity, INT echoBack)
{
	if ((mSePort = Serial_Open(port, baudrate)) > 0){
		return 0; //ok
	}
	else{
		return 1; //ng
	}
}

INT SerialLinux::Close(void)
{
	if (mSePort < 0){
		return -1;
	}

	Serial_Close(mSePort);

	return 0;
}

INT SerialLinux::Send(const void* data, INT bytes, UINT32 timeout)
{
	if (mSePort < 0){
		return -1;
	}

	if (Serial_Send(mSePort, data, bytes) > 0){
		return 0; //ok
	}
	else{
		return 1; //ng
	}
}

INT SerialLinux::Receive(void* data, INT bytes, UINT32 timeout)
{
	if (mSePort < 0){
		return -1;
	}

	return Serial_Receive(mSePort, data, bytes, timeout, mCB, mParam);
}

INT SerialLinux::Init()
{
	//nothing

	return 0;
}

INT SerialLinux::SettingEx(const UINT32 *pdwParam, UINT32 dwMask)
{
	//nothing

	return 0;
}

INT SerialLinux::SendCheck()
{
	//nothing

	return 0;
}

INT SerialLinux::ReceiveCheck()
{
	//nothing

	return 0;
}

INT SerialLinux::Clear()
{
	Serial_Clear(mSePort);

	return 0;
}

INT SerialLinux::Termination(INT iOn)
{
	//nothing

	return 0;
}

void SerialLinux::SetLoopCallback(LoopCallback func, void *param)
{
	mCB = func;
	mParam = param;
}

//-------------------------------------------------------------
/// @brief Open serial port
//-------------------------------------------------------------
/// @param port	serialPortDeviceName (ex.)/dev/ttyUSB0
/// @param br	baudrate(9600,19200,38400,57600,115200,230400,460800)
//-------------------------------------------------------------
/// @retval	portFD OK @n
///			<0     NG
//-------------------------------------------------------------
/// @note	startBit=1, stopBit=1, dataBits=8, parity=Even
//-------------------------------------------------------------
static int Serial_Open(const char *port, long br)
{
	int fd = open(port, O_RDWR | O_NONBLOCK);	// non blocking I/O

	if (fd < 0){
		return -1;
	}

	speed_t spd;

	switch (br){
	case 9600:
		spd = B9600;
		break;
	case 19200:
		spd = B19200;
		break;
	case 38400:
		spd = B38400;
		break;
	case 57600:
		spd = B57600;
		break;
	case 115200:
		spd = B115200;
		break;
	case 230400:
		spd = B230400;
		break;
	default:
		spd = B460800;
		break;
	}

	struct termios stio;
	memset(&stio, 0, sizeof(stio));
	stio.c_cflag = CS8 | CLOCAL | CREAD | PARENB;// Start=1, Stop=1, Data=8Bits, Parity=even
	stio.c_cc[VTIME] = 0;
	stio.c_cc[VMIN] = 0;
	cfsetispeed(&stio, spd);
	cfsetospeed(&stio, spd);
	tcsetattr(fd, TCSANOW, &stio);

	return fd;
}

//-------------------------------------------------------------
/// @brief Send serial
//-------------------------------------------------------------
/// @param port	 serialPortFD(FileDesciptor)
/// @param data  sendData
/// @param bytes sendByteLength
//-------------------------------------------------------------
/// @retval sendByteLength
//-------------------------------------------------------------
static int Serial_Send(int port, const void *data, int bytes)
{
	return write(port, data, bytes);
}

//-------------------------------------------------------------
/// @brief Calc elapsed time
//-------------------------------------------------------------
/// @param time	currentTime[timespec]
/// @param start startTime[timespec]
//-------------------------------------------------------------
/// @retval elapsedTime[ms]
//-------------------------------------------------------------
static long msElapse(struct timespec *time, struct timespec *start)
{
	long sec = time->tv_sec - start->tv_sec;
	long nsec = time->tv_nsec - start->tv_nsec;

	return (sec * 1000 + nsec / 1000000);
}

//-------------------------------------------------------------
/// @brief Receive serial
//-------------------------------------------------------------
/// @param port    serialPortFD(FileDesciptor)
/// @param data    receiveDataPointerAddress
/// @param bytes   receiveByteLength
/// @param timeout timeout[ms]
//-------------------------------------------------------------
/// @retval 0 OK
//-------------------------------------------------------------
static int Serial_Receive(int port, void *data, int bytes, unsigned int timeout, SerialLinux::LoopCallback cb, void *param)
{
	UINT8 *pb = (UINT8 *)data;
	int ret;
	bool isTimeout = false;

	// loop sleep time
	struct timespec ts;
	ts.tv_sec = 0;
	ts.tv_nsec = (long)(100e-6 / 1e-9);	// 100us

	// start check timeout
	struct timespec time, stime;
	clock_gettime(CLOCK_REALTIME, &stime);

	while (bytes > 0){
		ret = read(port, pb, bytes);
		if (ret < 0){
			if (errno == EAGAIN){ // In blocking mode, read<0 & errno=EAGEIN if there is no received data.
				ret = 0;
			}
			else{
				return -1;
			}
		}
		pb += ret;
		bytes -= ret;

		if (cb != NULL && bytes > 0){
			cb((SerialLinux *)param);
		}

		if (timeout != 0xFFFFFFFF){
			clock_gettime(CLOCK_REALTIME, &time);
			if (msElapse(&time, &stime) >= (long)timeout){
				isTimeout = true;
				break;
			}
			else{
				// do nothing
			}
		}
		else{
			// do nothing
		}

		nanosleep(&ts, NULL);
	}

	if (isTimeout){
		return 1;	//ng
	}

	return 0;
}

//-------------------------------------------------------------
/// @brief Close serial
//-------------------------------------------------------------
/// @param port	serialPortFD(FileDescriptor)
//-------------------------------------------------------------
static void Serial_Close(int port)
{
	close(port);
}

//-------------------------------------------------------------
/// @brief Clear buffer
//-------------------------------------------------------------
/// @param port	serialPortFD(FileDescriptor)
//-------------------------------------------------------------
static void Serial_Clear(int port)
{
	//TODO:
}

//int main(int argc,char **argv)
//{
//	int port = Serial_Open("/dev/ttyUSB0",115200);
//	printf("port %d\n",port);
//	if(port >= 0){
//		Serial_Send(port,"test",5);
//		Serial_Close(port);
//	}
//	return 0;
//}

