//---------------------------------------------------------------------------
///	@file
///		aspina_hand_node
///	@brief
///		ASPINA robotic gripper server node
///	@attention
///		Copyright (C) 2020 - SHINANO-KENSHI Co,Ltd
///	@author
///		7022
//---------------------------------------------------------------------------
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the name, the copyright and the trademark of the SHINANO-KENSHI
 *     CO.,LTD. nor the names of its contributors may be used to endorse or
 *     promote products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ros/ros.h>
#include <sstream>

#include "aspinaLib/Modbus.h"
#include "aspinaLib/HandModbus.h"
#include "aspinaLib/RobotARH.h"
#include "aspinaLib/serialLinux.hpp"

#include <aspina_hand/Hand.h>	// サービスによるハンド制御タイプ
#include <aspina_hand/AspinaHandCommand.h>	// ハンド制御コマンド・メッセージ
#include <aspina_hand/AspinaHandStatus.h>	// ハンド制御ステータス・メッセージ

#include "LibCmd.h"

#define	DEBUG_MODE	0	// 0:Release/1:Debug

// ローカル変数
static RobotARH *mRobotARH = NULL;
static HandModbus *mHandModbus = NULL;
static SerialLinux *mSerial = NULL;
static ros::Publisher mPub;
static int mHandNumber = 0;

// 静的関数宣言
static bool Callback(aspina_hand::Hand::Request &req, aspina_hand::Hand::Response &res);
static void HandCtrlCallback(const ros::MessageEvent<aspina_hand::AspinaHandCommand const> &event);
static void CommCallback(void *param);

// サービスコールバック
static bool Callback(aspina_hand::Hand::Request &req, aspina_hand::Hand::Response &res)
{
	int id = req.id;

	if (DEBUG_MODE == 1){
		printf("id : %d\n", (int)id);
	}
	if (id < 1 || id > mHandNumber){
		return false;
	}

	mRobotARH->setSlaveID(id);

	if (DEBUG_MODE == 1){
		printf("cmd %d\n", req.cmd);
	}
	if (req.cmd == 0){
		// 制御
		mRobotARH->setHandTime((int)req.runtime);
		if (DEBUG_MODE == 1){
			printf("go(%d,%d,%d)\r\n", req.position, req.effort, req.runtime);
		}
		if (mRobotARH->moveHand((int)req.position, (int)req.effort, false) != 0){
			printf("hand error1\n");
			return false;
		}
		else{
			res.position = 0;
			res.effort = 0;
			res.stalled = 0;
			res.reached_goal = 0;
		}
	}
	else{
		// 状態取得
		UINT32 status;
		if (mRobotARH->getHandStatus(&status) == 0){
			int position = mRobotARH->getHandCurrentPosition();
			int effort = mRobotARH->getHandCurrentTorque();
			if (effort < 0){
				effort = -effort;
			}
			res.position = position;
			res.effort = effort;
			if ((status & (HandStatusCode::bitSrvOn | HandStatusCode::bitHomeOk)) == (HandStatusCode::bitSrvOn | HandStatusCode::bitHomeOk)){
				if ((status & HandStatusCode::bitAlarm) != 0){
					res.stalled = 1;
				}
				else{
					res.stalled = 0;
				}
				if ((status & HandStatusCode::bitReady) != 0){
					res.reached_goal = 1;
				}
				else{
					res.reached_goal = 0;
				}
			}
			else{
				res.stalled = 0;
				res.reached_goal = 0;
			}
		}
		else{
			printf("hand error2\n");
			return false;
		}
	}

	return true;
}

// トピック受信コールバック(AspinaHandCommand)
static void HandCtrlCallback(const ros::MessageEvent<aspina_hand::AspinaHandCommand const> &event)
{
	int id = event.getMessage()->id;
	if (id >= 1 && id <= mHandNumber){
		aspina_hand::Hand::Request req;
		aspina_hand::Hand::Response res;
		req.cmd = event.getMessage()->cmd;
		req.id = id;
		req.position = event.getMessage()->position;
		req.effort = event.getMessage()->effort;
		req.runtime = event.getMessage()->runtime;
		bool ret = Callback(req, res);
		aspina_hand::AspinaHandStatus sts;
		sts.id = id;
		sts.position = 0;
		sts.effort = 0;
		sts.reached_goal = 0;
		if (ret == true){
			sts.result = 0;
			if (req.cmd == 0){
				sts.stalled = 0;
			}
			else{
				sts.position = res.position;
				sts.effort = res.effort;
				sts.reached_goal = res.reached_goal;
				sts.stalled = res.stalled;
			}
		}
		else{
			sts.result = 1;
			sts.position = 0;
			sts.effort = 0;
			sts.reached_goal = 0;
			sts.stalled = 1;
		}
		// ステータス・パブリッシュ
		mPub.publish(sts);
	}
	else{
		// do nothing
	}
}

// 通信コールバック
static void CommCallback(void *param)
{
	ros::Rate *rate = (ros::Rate *)param;

	rate->sleep();
}

// メイン
int main(int argc, char **argv)
{
	// パラメータ処理
	char commDevice[256];
	int hand_number = 1;
	strcpy(commDevice, "/dev/ttyUSB0");
	CmdOption option(argc, argv);
	bool illegal = false;
	for (;;){
		char *ret;
		bool isOption;
		ret = option.Get(&isOption);
		if (ret == NULL){
			break;
		}
		else{
			if (isOption == true){
				if (*ret == 'd' || *ret == 'D'){
					ret = option.Get(&isOption);
					if (ret == NULL || isOption == true){
						illegal = true;
						break;
					}
					else{
						option.strcpy_s(commDevice, sizeof(commDevice), ret);
					}
					continue;
				}
				else{
					// do nothing
				}
				if (*ret == 'h' || *ret == 'H'){
					ret = option.Get(&isOption);
					if (ret == NULL || isOption == true){
						illegal = true;
						break;
					}
					else{
						hand_number = atoi(ret);
						if (hand_number < 1 || hand_number > 254){
							illegal = true;
							break;
						}
						else{
							// do nothing
						}
					}
					continue;
				}
			}
			else{
				illegal = true;
				break;
			}
		}
	}
	if (illegal == true){
		printf("illegal command.\n");
		return -1;
	}

	printf("aspina_hand :\n");
	printf("  [comm. device : %s]\n", commDevice);
	printf("  [number of hand : %d]\n", hand_number);
	mHandNumber = hand_number;

	// ROS初期化
	argc = 1;
	ros::init(argc, argv, "aspina_hand");

	// サービス登録
	ros::NodeHandle n;
	ros::ServiceServer server = n.advertiseService("aspina_hand/hand", Callback);

	// パブリッシャ登録
	mPub = n.advertise<aspina_hand::AspinaHandStatus>("aspina_hand/status", 1);

	// サブスクライバ登録
	ros::Subscriber sub = n.subscribe("aspina_hand/cmd", 1, HandCtrlCallback);

	// 制御周期(1秒間に何サイクル回すか)
	// ※ノードハンドル（トピックなど）登録してから設定しないと例外が発生
	ros::Rate loop_rateMain(1000);		// メインループ用
	ros::Rate loop_rateCallback(1000);	// 通信コールバック用

	// 通信関連セットアップ
	mSerial = new SerialLinux();
	mSerial->SetLoopCallback(CommCallback, &loop_rateCallback);
	mHandModbus = new HandModbus(mSerial);
	mRobotARH = new RobotARH();
	mRobotARH->setIHandComm(mHandModbus);
	mRobotARH->setBaudrate(115200);
	mRobotARH->setSlaveID(1);
	if (mRobotARH->openComm(commDevice) != 0){
		printf("port(%s) open error\n", commDevice);
		return -1;
	}

	// メッセージ受付
	while (ros::ok()){
		ros::spinOnce();
		loop_rateMain.sleep();
	}

	return 0;
}

